capture log close
clear all
macro drop _all
version 17.0
cd "/Users/mike_cunningham/Documents/Stata/OutputWeights25"
set more off
set maxiter 16000
set iterlog off

log using Output/log/anLeonCost-DNSP25eiaa-22, replace text

/*--------------------------------------------------------------------------
Project: Quantonomics - DNSP output weights
Author:  Michael Cunningham 
File:    anLeonCost-DNSP25eiaa-22.do
Initial Date:    31-Aug-2024
Revised Date:	 19-Sep-2025
--------------------------------------------------------------------------*/

tempfile catvar matB

********* Import initial parameter values ******

import excel using Input/DNSPei-InitialValues.xlsx, cellrange(A1:G53) clear firstrow 

forvalues i = 1/13 {
	mkmat a1 a2 a3 a4 b if dnsp == `i', matrix(A`i')
}
gen obs = _n
keep obs dnsp input
save `catvar'

********* Import data ***************

import excel using "Input/DNSP consolidated benchmarking data (2024).xlsx", ///
	sheet("Shazam DNSP Data") cellrange(A1:V249) clear firstrow case(l)
drop in 1
destring, replace	

********* Data management ********		
		
rename dnspyearidentifier			  dnspyear
rename totalrevenue                   rev
rename energy						  y1	
rename customernumbers                y3	
rename maximumdemand                  md
rename ratchedmaximumdemand	          y2
rename circuitlength                  y4	
rename customerminutesoffsupply       cmos	
rename pricecustomerminutesoffsuppl   p_cmos 	
rename priceopex                      p_x1	
rename overheadsubtransmissionlines   ohsubt 	
rename overheaddistributionlinesund   ohdist
rename undergroundsubtransmissionline ugsubt
rename undergrounddistributionlines   ugdist
rename transformersexcludingfirststa  x4
rename aucoverheadsubtransmission     auc_ohsubt
rename aucoverheaddistribution        auc_ohdist
rename aucundergroundsubtransmission  auc_ugsubt
rename aucundergrounddistribution     auc_ugdist
rename auctransformersexcludingfirst  auc_transf
rename shareofundergroundcables		  ugshr

gen double x1 = opex/p_x1
gen double x2 = ohsubt + ohdist
gen double x3 = ugsubt + ugdist

gen double p_x2 = (auc_ohsubt + auc_ohdist) / x2
gen double p_x3 = (auc_ugsubt + auc_ugdist) / x3
gen double p_x4 = auc_transf / x4

gen int dnsp = int(dnspyear/100)
gen int year = dnspyear - dnsp*100 + 2000
replace cmos = 1 if cmos < 1
gen int t = year - 2005

local varlist y1 y2 y3 y4 x1 x2 x3 x4 p_x1 p_x2 p_x3 p_x4 t
	
format %14.0g `varlist'
sort dnsp year
keep dnsp year `varlist'
order dnsp year `varlist'
keep if year <= 2022

label variable x1 "Real Opex"
label variable x2 "OH/Lines"
label variable x3 "UG/Cables"
label variable x4 "Transformers"
label variable p_x1 "Price Opex"
label variable p_x2 "Price OH/Lines"
label variable p_x3 "Price UG/Cables"
label variable p_x4 "Price Transformers"
label variable y1 "Energy"
label variable y2 "RMD"
label variable y3 "Customer No"
label variable y4 "Circuit Length"
label variable t  "Time"
describe	

********* Estimate Leontief input demand models ****************

gen ct = 0						// total cost 

foreach k in 1 2 3 4 {
		
	forvalues m = 1/4 {
		gen v`k'_`m' = 0		// contribution of output m to input j cost
	}
	gen v`k' = 0				// contribution of output m to total cost
	gen hx`k' = 0               // predicted values of x1, x2, x3, x4
}	

matrix define A = J(4,1,0)

forvalues i = 1/13 {
	forvalues j = 1/4 {
		
		di "DNSP " `i' ", Input " `j'
		di "=========================="
		matrix b0 = A`i'[`j',1...]   // picks up relevant row of initial values matrix
		matrix list b0
		
		nl (x`j' = ({a1}^2*y1 + {a2}^2*y2 + {a3}^2*y3 + {a4}^2*y4) * (1+{b}*t)) ///
			if dnsp == `i', initial(b0)                       // equation (2.1)
		estimates store xdem_`i'_`j'
		di "Estimates stored as xdem_`i'_`j' "
		predict yhat_x`j'
		replace hx`j' = yhat_x`j' if dnsp == `i'
		drop yhat_x`j'
		
		scalar iter`i'_`j' = e(ic)
		scalar conv`i'_`j' = e(converged)
			
		matrix b`j' = e(b)          // vector of parameter estimates
		matrix drop b0		
		
		di "Looping across outputs"
		forvalues m = 1/4 {
			replace v`j'_`m' = p_x`j' * (_b[a`m':_cons])^2 * y`m' * ///
				(1 + _b[b:_cons]* t) if dnsp == `i'  // contribution of output m to input j cost				
			}    		
	}
	
	matrix B`i' = b1\b2\b3\b4
	matrix drop b1 b2 b3 b4
	matrix list B`i'
	
	replace v1 = v1_1 + v2_1 + v3_1 + v4_1 if dnsp == `i'      // contribution of output 1 to total cost
	replace v2 = v1_2 + v2_2 + v3_2 + v4_2 if dnsp == `i' 	   // contribution of output 2 to total cost
	replace v3 = v1_3 + v2_3 + v3_3 + v4_3 if dnsp == `i' 	   // contribution of output 3 to total cost
	replace v4 = v1_4 + v2_4 + v3_4 + v4_4 if dnsp == `i' 	   // contribution of output 4 to total cost
			
	replace ct = v1 + v2 + v3 + v4 if dnsp == `i'             // total cost 
			
	est tab xdem_`i'_1  xdem_`i'_2   xdem_`i'_3   xdem_`i'_4 , b(%9.3f) ///
		t(%9.2f) stats(r2) stfmt(%9.3f)
	putexcel set "Output/xlsx/LeonCostResults-DNSP25eiaa-22.xlsx", modify ///
		sheet(dnsp`i', replace)
	putexcel A1 = etable
	putexcel save
}

gen s1 = v1/ct
gen s2 = v2/ct 
gen s3 = v3/ct
gen s4 = v4/ct

summarize s1 s2 s3 s4 

forvalues k = 1/4 {
	qui summarize s`k'
	scalar s`k'_avg = r(mean)
	matrix A[`k',1] = s`k'_avg
}
matrix rownames A = output1 output2 output3 output4
matrix colnames A = s
matrix list A

putexcel set "Output/xlsx/LeonCostResults-DNSP25eiaa-22.xlsx", ///
	sheet("Cost shares", replace) modify
putexcel A1=matrix(A), names
putexcel save

forvalues i = 1/13 {
	forvalues j = 1/4 {
		di "Iterations: DNSP " `i' ", Input " `j' " = " iter`i'_`j'
	}	
}
forvalues i = 1/13 {
	forvalues j = 1/4 {
		di "Convergence: DNSP " `i' ", Input " `j' " = " conv`i'_`j'
	}	
}

drop _est*
export excel using "Output/xlsx/LeonCostResults-DNSP25eiaa-22.xlsx", firstrow(var) ///
	sheet("fulldata", replace)
	
log close
exit

	